/************************************************************************
 *
 * \file: GadgetLoader.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Baidu Carlife - Demo application
 *
 * \author: Ajay Kumar Sahoo / ajaykumar.sahoo@in.bosch.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <adit_logging.h>
#include <utils/GadgetLoader.h>
#include <string.h>
#include <glob.h>
#include <libkmod.h>
#include <sys/utsname.h>
#include <sys_time_adit.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <errno.h>

LOG_IMPORT_CONTEXT(tbdcl)

namespace adit { namespace bdcl {
GadgetLoader::GadgetLoader()
{
    _udcParam = nullptr;
}
GadgetLoader::~GadgetLoader()
{

}
//only When IOS device is used. On shutdown Unload Gadget and switch HU USB Otg to HOST
int32_t GadgetLoader::switchHUToHost()
{
    int32_t res;

    //unload all gadget modules and reset otg role to host
    UnloadGadgetModules();

    /* turn off VBUS power */
    res = iap2SwitchVbusPower("off");
    printf(" %u ms  iap2SwitchVbusPower(off)  = %d \n",iap2CurrTimeMs(), res);

    sleep(2);

    /* switch the USB OTG port to 'gadget' */
    res = iap2SwitchOTG(STR_HOST);
    printf(" %u ms  iap2SwitchOTG(%s)  = %d \n",iap2CurrTimeMs(), STR_HOST, res);


    /* turn on VBUS power */
    res = iap2SwitchVbusPower("on");
    printf(" %u ms  iap2SwitchVbusPower(on)  = %d \n",iap2CurrTimeMs(), res);


    return res;
}
int32_t GadgetLoader::switchHUToGadget()
{
    int32_t res=0;

    if (_udcParam == nullptr)
    {
        _udcParam = new udcParamInfo_t;
        memset(_udcParam, 0, sizeof(udcParamInfo_t));
    }

    /* switch the USB OTG port to 'gadget' */
    res = iap2SwitchOTG(STR_GADGET);
    LOG_INFO((tbdcl, "Server::%s()  %u ms  iap2SwitchOTG(%s)  = %d",
            __func__,iap2CurrTimeMs(), STR_GADGET, res));

    if (0 == res) {
        /* turn on VBUS power */
        res = iap2SwitchVbusPower("on");
        LOG_INFO((tbdcl, "Server::%s()  %u ms  iap2SwitchVbusPower(on)  = %d", __func__, iap2CurrTimeMs(), res));
    }

    if (0 == res) {
        res = LoadGadgetModules();
        LOG_INFO((tbdcl, "Server::%s()  %u ms  LoadGadgetModules()  = %d", __func__, iap2CurrTimeMs(), res));
    }

    return res;
}

int32_t GadgetLoader::LoadGadgetModules()
{
    int32_t res = 0;
    char *mount_fs_param = nullptr;

    if (isKernel314())
    {
        res = LoadKernelModule(NULL, CONFIGFS_MODULE_NAME, strlen(CONFIGFS_MODULE_NAME));
    }

    if (res == 0)
    {
        res = mount("config", "/sys/kernel/config", "configfs", MS_NOEXEC, mount_fs_param);
        if (res != 0)
        {
            LOG_ERROR((tbdcl,"CTL ERROR"));
        }
        LOG_INFO((tbdcl," %u ms  mount() res = %d\n", iap2CurrTimeMs(), res));

        res = chmod("/sys/kernel/config", 0750);
        if (res != 0)
        {
            LOG_ERROR((tbdcl,"CTL ERROR"));
        }
        LOG_INFO((tbdcl, " %u ms  chmod() res = %d\n", iap2CurrTimeMs(), res));
    }

    if(res == 0) {
        if(0 != LoadKernelModule(NULL, LIBCOMPOSITE_MODULE_NAME, strlen(LIBCOMPOSITE_MODULE_NAME))) {
            res = -1;
        }
    }

    memset(&usb_gadget_configuration, 0, sizeof(iAP2_usbg_config_t));
    if(res  == 0)
    {
        /* Initialize - 1st Gadget Configuration */
        res = iap2InitGadgetConfiguration(&usb_gadget_configuration, 1);
        if(res == 0)
        {
            res = iAP2InitializeGadget(&usb_gadget_configuration);
            LOGD_DEBUG((tbdcl,"iAP2InitializeGadget = %d\n", res ));
            LOG_INFO((tbdcl, "iAP2InitializeGadget rc = %d ", res));
        }
    }

    if(res == 0)
    {
        res = mkdir(FUNCTION_FS_PATH, 0777);
        if (res != 0){
            res = -1;
            LOG_INFO((tbdcl, "mkdir function fs = %d ", res));
        }

        mount_fs_param = nullptr;

        res = mount("ffs_1", FUNCTION_FS_PATH, FUNCTION_FS_TYPE, MS_NOEXEC, mount_fs_param);
        if (res != 0){
            res = -1;
            LOG_INFO((tbdcl, "mount function fs = %d errno: %d", res, errno));
        }

        /* after mount we need to set the mode again */
        res = chmod(FUNCTION_FS_PATH, 0777);
        if (res != 0){
            res = -1;
            LOG_INFO((tbdcl, "chmod function fs = %d errno: %d", res, errno));
        }

        res = chmod("/dev/ffs/ep0", 0777);
        if (res != 0){
            res = -1;
            LOG_INFO((tbdcl, "chmod function fs ep0 = %d errno: %d", res, errno));
        }
    }

    if(res == 0)
    {
        uint8_t initEndPoint[] =  {"/dev/ffs/ep0"};

        usb_gadget_configuration.iAP2FFSConfig.initEndPoint = (uint8_t*)strdup((const char*)initEndPoint);
        res = iAP2InitializeFFSGadget(&usb_gadget_configuration);
        LOG_INFO((tbdcl, "InitializeFFSGadget = %d ", res));
    }

    return res;
}
int32_t GadgetLoader::UnloadGadgetModules()
{
    int res = -1;

    if(0 != umount("/sys/kernel/config"))
    {
        printf("umount config errno:%d-%s \n", errno,strerror(errno));
    }else {
        rmdir("/sys/kernel/config");
    }

    if(0 != umount(FUNCTION_FS_PATH))
    {
        printf("umount ffs errno:%d-%s \n", errno,strerror(errno));
    }
    else
    {
        res = rmdir(FUNCTION_FS_PATH);
        printf("rmdir res : %d  \n", res);
    }


    if (isKernel314())
    {
        res = UnloadKernelModule(USB_F_FS_MODULE_NAME);
        res = UnloadKernelModule(LIBCOMPOSITE_MODULE_NAME);
        res = UnloadKernelModule(CONFIGFS_MODULE_NAME);
    }

    return res ;
}
//int GadgetLoader::UnloadGadgetModules(const char* modname)
int32_t GadgetLoader::UnloadKernelModule(const char* modname)
{
    int rc = -1;
    struct kmod_ctx *ctx;
    struct kmod_module *module;

    ctx = kmod_new(NULL, NULL);
    if (ctx == NULL){
        printf("iap2UnloadModule()  kmod_new() failed\n");
        rc = -1;
    } else{
        rc = kmod_module_new_from_name(ctx, modname, &module);
        if (rc){
            printf("iap2UnloadModule()  kmod_module_new_from_name() failed\n");
            rc = -1;
        } else{
            printf("iap2UnloadModule()  modname='%s' obj=%p\n", modname, module);

            rc = kmod_module_remove_module(module, KMOD_REMOVE_FORCE);
            if (0 == rc){
                fprintf(stderr, " %u ms  iap2UnloadModule()  unloaded module %s\n",
                        iap2CurrTimeMs(), kmod_module_get_name(module));
            } else if (rc == KMOD_PROBE_APPLY_BLACKLIST){
                fprintf(stderr, "iap2UnloadModule()  module %s black listed\n",
                        kmod_module_get_name(module));
            } else{
                fprintf(stderr, "iap2UnloadModule()  failed to remove module %s | err = %d \n",
                        kmod_module_get_name(module), rc);
            }
            kmod_module_unref(module);
        }
        kmod_unref(ctx);
    }

    return rc;
}

void GadgetLoader::setRollSwitchInfo(std::string iap2devserial)
{
    /* PRQA: Lint Message 689:  IAP2_USB_ROLE_SWITCH_OTG_GLOB_314 is a useful and valid string */
    /*lint -save -e689*/

    roleSwitchInfo.vendorId = APPLE_VENDOR_ID;
    roleSwitchInfo.productId = APPLE_PRODUCT_ID_MIN;
    roleSwitchInfo.serialNumber = (char*)iap2devserial.c_str();
    roleSwitchInfo.mode = IAP2_USB_ROLE_SWITCH_WITHOUT_DIGITAL_IPOD_OUT; //no iOS in the car
    roleSwitchInfo.otgGlob = (char*)IAP2_USB_ROLE_SWITCH_OTG_GLOB_314;

    /*lint -restore*/
}


int32_t GadgetLoader::LoadKernelModule(const iap2LoadModuleParameters* iap2ModuleLoadParam, const char* modname, uint32_t length)
{
    int rc = -1;
    struct kmod_ctx *ctx;
    struct kmod_module *module;

    ctx = kmod_new(NULL, NULL);
    if (ctx == NULL){
        printf("iap2LoadModule()  kmod_new() failed\n");
        rc = -1;
    } else{
        rc = kmod_module_new_from_name(ctx, modname, &module);
        if (rc){
            printf("iap2LoadModule()  kmod_module_new_from_name() failed\n");
            rc = -1;
        } else{
            printf("iap2LoadModule()  modname='%s' obj=%p\n", modname, module);

            if (strncmp(modname, CONFIGFS_MODULE_NAME, length) == 0){
                if (iap2ModuleLoadParam != NULL) {
                    printf ("iap2LoadModule()   No parameters supported for module %s\n", kmod_module_get_name(module));
                    return -1;
                }
                rc = kmod_module_insert_module(module, KMOD_PROBE_APPLY_BLACKLIST, NULL);
                if (0 == rc){
                    printf(" %u ms  iap2LoadModule()  loaded module %s\n",
                            iap2CurrTimeMs(), kmod_module_get_name(module));
                } else if (rc == KMOD_PROBE_APPLY_BLACKLIST){
                    printf("iap2LoadModule()  module %s black listed\n",
                            kmod_module_get_name(module));
                } else{
                    printf("iap2LoadModule()  failed to insert module %s | err = %d \n",
                            kmod_module_get_name(module), rc);
                }

            } else if (strncmp(modname, LIBCOMPOSITE_MODULE_NAME, length) == 0){

                if (iap2ModuleLoadParam != NULL) {
                    printf ("iap2LoadModule()   No parameters supported for module %s\n", kmod_module_get_name(module));
                    return -1;
                }

                rc = kmod_module_insert_module(module, KMOD_PROBE_APPLY_BLACKLIST, NULL);
                if (0 == rc){
                    printf(" %u ms  iap2LoadModule()  loaded module %s\n",
                            iap2CurrTimeMs(), kmod_module_get_name(module));
                } else if (rc == KMOD_PROBE_APPLY_BLACKLIST){
                    printf("iap2LoadModule()  module %s black listed\n",
                            kmod_module_get_name(module));
                } else{
                    printf("iap2LoadModule()  failed to insert module %s | err = %d \n",
                            kmod_module_get_name(module), rc);
                }
            } else if (strncmp(modname, USB_F_FS_MODULE_NAME, length) == 0){
                if (iap2ModuleLoadParam != NULL) {
                    printf ("iap2LoadModule()   No parameters supported for module %s\n", kmod_module_get_name(module));
                    return -1;
                }

                rc = kmod_module_insert_module(module, KMOD_PROBE_APPLY_BLACKLIST, NULL);
                if (0 == rc){
                    printf(" %u ms  iap2LoadModule()  loaded module %s\n",
                            iap2CurrTimeMs(), kmod_module_get_name(module));
                } else if (rc == KMOD_PROBE_APPLY_BLACKLIST){
                    printf("iap2LoadModule()  module %s black listed\n",
                            kmod_module_get_name(module));
                } else{
                    printf("iap2LoadModule()  failed to insert module %s | err = %d \n",
                            kmod_module_get_name(module), rc);
                }
            } else if (strncmp(modname, U_ETHER_MODULE_NAME, length) == 0){
                if (iap2ModuleLoadParam != NULL) {
                    printf ("iap2LoadModule()   No parameters supported for module %s\n", kmod_module_get_name(module));
                    return -1;
                }

                rc = kmod_module_insert_module(module, KMOD_PROBE_APPLY_BLACKLIST, NULL);
                if (0 == rc){
                    printf(" %u ms  iap2LoadModule()  loaded module %s\n",
                            iap2CurrTimeMs(), kmod_module_get_name(module));
                } else if (rc == KMOD_PROBE_APPLY_BLACKLIST){
                    printf("iap2LoadModule()  module %s black listed\n",
                            kmod_module_get_name(module));
                } else{
                    printf("iap2LoadModule()  failed to insert module %s | err = %d \n",
                            kmod_module_get_name(module), rc);
                }
            } else{
                /* error. unknown module name */
                printf("iap2LoadModule()  unknown module name\n");
                rc = -1;
            }
            kmod_module_unref(module);
        }
        kmod_unref(ctx);
    }

    return rc;
}

/* unload kernel module */

uint32_t GadgetLoader::iap2CurrTimeValToMs(struct timeval* currTime)
{
    return (uint32_t)(currTime->tv_sec * 1000) + (uint32_t)(currTime->tv_usec / 1000);
}

uint32_t GadgetLoader::iap2CurrTimeMs(void)
{
    uint32_t timeMs;
    struct timeval tp;
    gettimeofday (&tp, NULL);
    timeMs = iap2CurrTimeValToMs(&tp);

    return timeMs;
}

/* return TRUE (1) if 3.14 Kernel. Otherwise FALSE (0). */
bool GadgetLoader:: isKernel314(void)
{
    struct utsname buf;
    if (0 == uname(&buf)) {
        if(0 == strncmp(buf.release, "3.14", strlen("3.14")))
            return true;
    }
    return false;
}
int32_t GadgetLoader::iap2FindVbus(void)
{
    int32_t rc = 0;
    glob_t found;

    /* find VBUS path */
    if (0 == (rc = glob(IAP2_VBUS_POWER_BDCL, 0, NULL, &found)) && found.gl_pathc > 0)
    {
        if (found.gl_pathc > 1)
        {
            printf("more than one VBUS found; \n\t\t use: %s \n",
                   found.gl_pathv[0]);
        }

        roleSwitchInfo.vbusPower = (char*)malloc(strlen(found.gl_pathv[0]) +1);
        strncpy(roleSwitchInfo.vbusPower, found.gl_pathv[0], strlen(found.gl_pathv[0]) +1);

        globfree(&found);
    } else if (rc == GLOB_NOMATCH){
        printf("glob does not found %s  rc = %d\n", IAP2_VBUS_POWER_BDCL, rc);
        rc = IAP2_CTL_ERROR;
    } else{
        printf("glob failed with rc = %d\n", rc);
        rc = IAP2_CTL_ERROR;
    }

    return rc;
}

int32_t GadgetLoader::iap2SwitchOTG( const char* value)
{
    int32_t status;
    char otgPath[STR_MAX_LENGTH];

    status = iap2findOTGPath(otgPath, STR_MAX_LENGTH, roleSwitchInfo.otgGlob);
    /* error logged and handled */

    if (status == IAP2_OK)
    {
        /* change VBUS auto behavior dependent on USB OTG mode.
         * host-to-gadget: we have to disable vbus_auto before setting role to gadget. */
        const size_t len = strlen(value) + 1;
        if (0 == strncmp(STR_GADGET, value, len - 1))
        {
            /* disable when switching to gadget. */
            status = iap2SwitchVbusAuto(otgPath, STR_DISABLE);
            printf(" %u ms  iap2SwitchVbusAuto(%s)  = %d \n",
                    iap2CurrTimeMs(), STR_DISABLE, status);
        }

        /* change USB OTG role */
        if ((status == IAP2_OK)
            && (TRUE != iap2CommonWrite(otgPath, "role", value, TRUE)))
        {
            printf("iap2SwitchOTG():  set USB OTG %s/role to %s failed \n", otgPath, value);
            status = IAP2_CTL_ERROR;
        }
        /* change VBUS auto behavior dependent on USB OTG mode.
         * gadget-to-host: we have to enable vbus_auto after setting role to host. */
        if (0 == strncmp(STR_HOST, value, len - 1))
        {
            /* enable when switching to host. */
            status = iap2SwitchVbusAuto(otgPath, STR_ENABLE);
            printf(" %u ms  iap2SwitchVbusAuto(%s)  = %d \n",
                   iap2CurrTimeMs(), STR_ENABLE, status);
            /* The host controller for the OTG port is not available after switching the role to gadget.
             * Therefore, vbus_auto does not exists until switching back the role to host. */
            /* Do not handle this as an error. */
            status = IAP2_OK;
        }
    }
    else
    {
        printf("iap2SwitchOTG():  did not find otgPath based on otgGlob: %s \n", roleSwitchInfo.otgGlob);
    }

    return status;
}
int32_t GadgetLoader::iap2SwitchVbusAuto(const char* otgPath, const char* value)
{
    int32_t ret = IAP2_OK;
    glob_t found;
    char valuePath[STR_MAX_LENGTH];
    char vbusAutoPath[STR_MAX_LENGTH];
    int file = 0;

    // with newly Kernel 3.8 patches and Kernel 3.14,
    // there could be an additional sub-path /usb*/ to the vbus_auto
    // e.g. /sys/devices/soc0/soc.0/21*/2184*/ci*/vbus_aauto
    //      or /sys/devices/soc0/soc.0/21*/2184*/ci*/usb*/vbus_auto

    // otgPath should be:   /sys/devices/soc0/soc.0/21*/2184*/ci*/
    ret = snprintf(valuePath, STR_MAX_LENGTH, "%s/%s/%s", otgPath, STR_USB_DYN_NUM, STR_VBUS_AUTO);
    if (ret >= 0 && ret < STR_MAX_LENGTH)
    {
        /* find VBUS_AUTO path */
        if (0 == (ret = glob(valuePath, 0, NULL, &found)) && found.gl_pathc > 0)
        {
            if (found.gl_pathc > 1)
            {
                printf("iap2SwitchVbusAuto()  more than one VBUS found; \n\t\t use: %s \n",
                       found.gl_pathv[0]);
            }
            strncpy(&vbusAutoPath[0], found.gl_pathv[0], strlen(found.gl_pathv[0]) +1);
            printf(" %u ms  iap2SwitchVbusAuto()  found vbus_auto=%s \n", iap2CurrTimeMs(), vbusAutoPath);
            ret = IAP2_OK;

            globfree(&found);
        } else if (ret == GLOB_NOMATCH){
            printf("iap2SwitchVbusAuto()  glob does not found %s  ret = %d\n", valuePath, ret);
            ret = IAP2_CTL_ERROR;
        } else{
            printf("iap2SwitchVbusAuto()  glob failed with ret = %d\n", ret);
            ret = IAP2_CTL_ERROR;
        }
        if (IAP2_OK != ret)
        {
            memset(valuePath, 0, STR_MAX_LENGTH);
            ret = snprintf(valuePath, STR_MAX_LENGTH, "%s/%s", otgPath, STR_VBUS_AUTO);
            if (ret >= 0 && ret < STR_MAX_LENGTH)
            {
                /* find VBUS_AUTO path */
                if (0 == (ret = glob(valuePath, 0, NULL, &found)) && found.gl_pathc > 0)
                {
                    if (found.gl_pathc > 1)
                    {
                        printf("iap2SwitchVbusAuto()  more than one VBUS found; \n\t\t use: %s \n",
                               found.gl_pathv[0]);
                    }

                    strncpy(&vbusAutoPath[0], found.gl_pathv[0], strlen(found.gl_pathv[0]) +1);
                    printf(" %u ms  iap2SwitchVbusAuto()  found vbus_auto=%s \n", iap2CurrTimeMs(), vbusAutoPath);
                    ret = IAP2_OK;

                    globfree(&found);
                } else if (ret == GLOB_NOMATCH){
                    printf("iap2SwitchVbusAuto()  glob does not found %s  ret = %d\n", valuePath, ret);
                    ret = IAP2_CTL_ERROR;
                } else{
                    printf("iap2SwitchVbusAuto()  glob failed with ret = %d\n", ret);
                    ret = IAP2_CTL_ERROR;
                }
            } else {
                printf("iap2SwitchVbusAuto()  create path to usb vbus_auto failed. \n");
                ret = IAP2_CTL_ERROR;
            }
        }
    } else {
        printf("iap2SwitchVbusAuto()  create path to vbus_auto failed. \n");
        ret = IAP2_CTL_ERROR;
    }

    if (ret == IAP2_OK)
    {
        file = open(vbusAutoPath, O_WRONLY);
        if (file >= 0)
        {
            close(file);
            if (TRUE != iap2CommonWriteValue(vbusAutoPath, value))
            {
                printf("iap2SwitchVbusAuto()  set %s to %s failed.\n",
                        vbusAutoPath, value);
                ret = IAP2_CTL_ERROR;
            }
        }
        else
        {
            if (ENOENT == errno)
            {
                printf("iap2SwitchVbusAuto()  open failed. %s does not exist.\n",
                        vbusAutoPath);
                /* currently, it's ok, because we don't know if the path is available */
                ret = IAP2_OK;
            }
            else
            {
                printf("iap2SwitchVbusAuto()  open failed: %d %s \n", errno, strerror(errno));
                ret = IAP2_CTL_ERROR;
            }
        }
    }
    else
    {
        printf("iap2SwitchVbusAuto()  create path to vbus_auto failed. \n");
        ret = IAP2_CTL_ERROR;
    }

    return ret;
}

int32_t GadgetLoader::iap2SwitchVbusPower(const char* value)
{
    int32_t status = IAP2_OK;

    if(NULL != roleSwitchInfo.vbusPower)
    {
        if (TRUE != iap2CommonWriteValue(roleSwitchInfo.vbusPower, value))
        {
            printf("iap2SwitchVbusPower():  set USB power failed \n");
            status = IAP2_CTL_ERROR;
        }
    }

    return status;
}

int32_t GadgetLoader::iap2findOTGPath(char* otgPath, U32 len, const char* otgGlob)
{

    strncpy(otgPath, otgGlob, len);
    strncat(otgPath, "/" STR_GADGET, len - strlen(otgPath));

    glob_t found;
    int32_t ret;
    if (0 == (ret = glob(otgPath, 0, NULL, &found)) && found.gl_pathc > 0)
    {
        strncpy(otgPath, found.gl_pathv[0], len);
        // otgPath is:  /sys/devices/soc0/soc.0/21*/2184*/ci*/gadget/

        /* terminate string STR_GADGET and '/' */
        otgPath[strlen(otgPath) - 7] = 0;
        // otgPath is:  /sys/devices/soc0/soc.0/21*/2184*/ci*/

        if (found.gl_pathc > 1)
        {
            printf("_iap2findOTGPath():  more than one OTG device found; use: %s \n", otgPath);
        }

        globfree(&found);
    }
    else if (ret == GLOB_NOMATCH)
    {
        printf("_iap2findOTGPath():  could not find USB OTG gadget at %s \n", otgPath);
        ret = IAP2_CTL_ERROR;
    }
    else
    {
        printf("_iap2findOTGPath():  USB OTG switch internal error: glob = %d \n", ret);
        ret = IAP2_CTL_ERROR;
    }

    return ret;
}

bool GadgetLoader::iap2CommonWrite(const char* path, const char* subPath, const char* value, bool checkBeforeWrite)
{
    bool status = FALSE;
    char valuePath[STR_MAX_LENGTH];
    int file = 0;

    int ret = snprintf(valuePath, STR_MAX_LENGTH, "%s/%s", path, subPath);
    if (ret >= 0 && ret < STR_MAX_LENGTH)
    {
        file = open(valuePath, checkBeforeWrite ? O_RDWR : O_WRONLY);
        if (file >= 0)
        {
            const size_t len = strlen(value) + 1;

            /* check before writing the same value */
            if (checkBeforeWrite == TRUE)
            {
                char buffer[len + 1]; /* to capture longer entries */
                memset(&buffer[0], 0, len + 1);
                ret = read(file, buffer, sizeof(buffer));
                if (ret == (int)len)
                {
                    if (0 == strncmp(buffer, value, len - 1)) /* without trailing \0 */
                    {
                        /* no need to write */
                        status = TRUE;
                    }
                    printf("_iap2CommonWrite():  read: %s \n", buffer);
                }
                else if (ret < 0)
                {
                    printf("_iap2CommonWrite():  read: %d %s \n", errno, strerror(errno));
                }
            }

            /* write or skip if already the same value */
            if (status == FALSE)
            {
                ret = write(file, value, len);
                if (ret == (int)len)
                {
                    /* successful write */
                    status = TRUE;
                }
                else if (ret < 0)
                {
                    printf("_iap2CommonWrite():  write: %d %s \n", errno, strerror(errno));
                }
            }

            close(file);
        }
        else
        {
            printf("_iap2CommonWrite():  open: %d %s \n", errno, strerror(errno));
        }
    }

    return status;
}

bool GadgetLoader::iap2CommonWriteValue(const char* path, const char* value)
{
    bool status = FALSE;
    int file = 0;
    int ret = 0;

    file = open(path, O_WRONLY);
    if (file >= 0)
    {
        const size_t len = strlen(value) + 1;

        ret = write(file, value, len);
        if (ret == (int)len)
        {
            /* successful write */
            status = TRUE;
        }
        else if (ret < 0)
        {
            printf("_iap2CommonWriteValue():  write: %d %s", errno, strerror(errno));
        }

        close(file);
    }
    else
    {
        printf("_iap2CommonWriteValue():  open: %d %s", errno, strerror(errno));
    }

    return status;
}

int32_t GadgetLoader::iap2InitGadgetConfiguration(iAP2_usbg_config_t* usb_gadget_configuration, uint8_t DeviceInstance)
{
    int32_t rc = 0;
    uint8_t iAP2GadgetName[STR_MAX_LENGTH - 1]         = {"iAP_Interface_"};
    uint8_t iAP2FFS_InstanceName[STR_MAX_LENGTH - 5]   = {"ffs_"};

    (void)snprintf((char*)&iAP2GadgetName[strlen((const char*)iAP2GadgetName)],
            ( sizeof(iAP2GadgetName) - strlen((const char*)iAP2GadgetName) ),
            "%d",
            DeviceInstance);
    (void)snprintf((char*)&iAP2FFS_InstanceName[strlen((const char*)iAP2FFS_InstanceName)],
            ( sizeof(iAP2FFS_InstanceName) - strlen((const char*)iAP2FFS_InstanceName) ),
            "%d",
            DeviceInstance);
    usb_gadget_configuration->iAP2GadgetName                = (uint8_t*)strndup((const char*)iAP2GadgetName, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2FFS_InstanceName          = (uint8_t*)strndup((const char*)iAP2FFS_InstanceName, STR_MAX_LENGTH - 4);
    usb_gadget_configuration->iAP2AccessoryName             = (uint8_t*)strndup((const char*)STR_ACC_INFO_NAME, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2AccessoryModelIdentifier  = (uint8_t*)strndup((const char*)STR_ACC_INFO_MODEL_IDENTIFIER, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2AccessoryManufacturer     = (uint8_t*)strndup((const char*)STR_ACC_INFO_MANUFACTURER, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2AccessorySerialNumber     = (uint8_t*)strndup((const char*)STR_ACC_INFO_SERIAL_NUM, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2AccessoryVendorId         = (uint8_t*)strndup((const char*)STR_ACC_INFO_VENDOR_ID, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2AccessoryProductId        = (uint8_t*)strndup((const char*)STR_ACC_INFO_PRODUCT_ID, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2AccessoryBcdDevice        = (uint8_t*)strndup((const char*)STR_ACC_INFO_BCD_DEVICE, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2ConfigFS_MountLocation    = (uint8_t*)strndup((const char*)CONFIGFS_MOUNT_LOCATION, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->iAP2UdcDeviceName             = (uint8_t*)strndup((const char*)UDC_DEVICE_NAME_IMX6, (STR_MAX_LENGTH - 1) );
    usb_gadget_configuration->CarPlayEnabled                = false;
    usb_gadget_configuration->iAP2_UAC_NCM_gadgets_disable  = true;

    if( (usb_gadget_configuration->iAP2GadgetName == NULL)                  ||
            (usb_gadget_configuration->iAP2FFS_InstanceName == NULL)            ||
            (usb_gadget_configuration->iAP2AccessoryName == NULL)               ||
            (usb_gadget_configuration->iAP2AccessoryModelIdentifier == NULL)    ||
            (usb_gadget_configuration->iAP2AccessoryManufacturer == NULL)       ||
            (usb_gadget_configuration->iAP2AccessorySerialNumber == NULL)       ||
            (usb_gadget_configuration->iAP2AccessoryVendorId == NULL)           ||
            (usb_gadget_configuration->iAP2AccessoryProductId == NULL)          ||
            (usb_gadget_configuration->iAP2AccessoryBcdDevice == NULL)          ||
            (usb_gadget_configuration->iAP2ConfigFS_MountLocation == NULL)      ||
            (usb_gadget_configuration->iAP2UdcDeviceName == NULL) )
    {
        rc = IAP2_ERR_NO_MEM;
        LOGD_DEBUG((tbdcl, "iap2InitGadgetConfiguration() - Not Enough Memory \n"));
    }

//    // fixme UAC isn't being used by BDCL but current iAP2 implementation doesn't work when we don't load UAC or NCM
//    // needs to be fixed by the ipod team
//    if (rc == 0) {
//        uint8_t iAP2UAC2_InstanceName[STR_MAX_LENGTH - 6] = "uac2";
//        char c_srate[] = {"44100,48000"};
//        char p_srate[] = {"44100,48000"};
//
//        usb_gadget_configuration->iAP2UAC2_InstanceName = (uint8_t*) strndup((const char*) iAP2UAC2_InstanceName,
//                STR_MAX_LENGTH - 5);
//        usb_gadget_configuration->iAP2_UAC2_Attrs = (usbg_f_uac2_attrs*)calloc(1, sizeof(usbg_f_uac2_attrs));
//        if ((usb_gadget_configuration->iAP2_UAC2_Attrs == NULL)
//                || (usb_gadget_configuration->iAP2UAC2_InstanceName == NULL)) {
//            rc = IAP2_ERR_NO_MEM;
//            LOGD_DEBUG((tbdcl, "%s() ERROR: Not Enough Memory", __FUNCTION__));
//        } else {
//            usb_gadget_configuration->iAP2_UAC2_Attrs->c_chmask = 3;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->c_srate_def = 44100;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->c_ssize = 2;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->delay_tout = 80;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->p_chmask = 0;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->p_srate_def = 44100;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->p_ssize = 2;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->c_srate = c_srate;
//            usb_gadget_configuration->iAP2_UAC2_Attrs->p_srate = p_srate;
//        }
//    }

    /* FFS configuration is done hardcoded just to make it run */
    if(rc == 0)
    {
        usb_gadget_configuration->iAP2FFSConfig.iOSAppNames = (uint8_t**)(calloc(1, sizeof(uint8_t*)));
        usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier = (uint8_t*)(calloc(1, sizeof(uint8_t)));
        if( (usb_gadget_configuration->iAP2FFSConfig.iOSAppNames == NULL) ||
                (usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier == NULL) )
        {
            rc = IAP2_ERR_NO_MEM;
        }
    }
    if( (rc == 0) )
    {
        usb_gadget_configuration->iAP2FFSConfig.nativeTransport = TRUE;
        usb_gadget_configuration->iAP2FFSConfig.iOSAppCnt = 1;
        usb_gadget_configuration->iAP2FFSConfig.iOSAppNames[0] = (uint8_t*)strdup("com.baidu.CarLifeVehicleProtocol");
        if(usb_gadget_configuration->iAP2FFSConfig.iOSAppNames[0] == NULL)
        {
            rc = IAP2_ERR_NO_MEM;
        }
        else
        {
            usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier[0] = 1;
        }
    }

    LOGD_DEBUG((tbdcl,"iap2InitGadgetConfiguration() - returns rc = %d \n", rc ));

    return rc;
}

}}
